<?php

namespace App\Http\Controllers\Siswa;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\PenggunaanBahan;
use App\Models\PenggunaanBahanDetail;
use App\Models\BahanPraktik;
use App\Models\Guru;

class SiswaPenggunaanBahanController extends Controller
{
    public function index()
    {
        if (!session('siswa_id')) return redirect()->route('login');

        $penggunaan = PenggunaanBahan::with('guru')
            ->where('id_siswa', session('siswa_id'))
            ->latest()
            ->get();

        return view('siswa.penggunaan_bahan.index', compact('penggunaan'));
    }

    public function create()
    {
        if (!session('siswa_id')) return redirect()->route('login');

        $bahan = BahanPraktik::where('stok', '>', 0)->get();
        $guru  = Guru::orderBy('nama')->get();

        return view('siswa.penggunaan_bahan.create', compact('bahan', 'guru'));
    }

    public function store(Request $r)
    {
        if (!session('siswa_id')) return redirect()->route('login');

        if (empty($r->bahan_id)) {
            return back()->with('error', 'Pilih minimal satu bahan.');
        }

        DB::beginTransaction();

        try {
            // 🔒 VALIDASI STOK & LIMIT
            foreach ($r->bahan_id as $i => $bahanId) {
                $jumlah = (float) ($r->jumlah[$i] ?? 0);

                if ($jumlah <= 0 || $jumlah > 10) {
                    throw new \Exception('Jumlah bahan maksimal 10 per item.');
                }

                $bahan = BahanPraktik::lockForUpdate()->findOrFail($bahanId);

                if ($bahan->stok < $jumlah) {
                    throw new \Exception("Stok {$bahan->nama} tidak mencukupi.");
                }
            }

            // 🧾 HEADER
            $penggunaan = PenggunaanBahan::create([
                'kode_penggunaan' => 'PB-' . date('YmdHis'),
                'id_siswa' => session('siswa_id'),
                'id_guru'  => $r->id_guru,
                'tanggal'  => $r->tanggal,
                'keterangan' => $r->keterangan
            ]);

            // 📦 DETAIL + POTONG STOK
            foreach ($r->bahan_id as $i => $bahanId) {
                $jumlah = (float) $r->jumlah[$i];

                PenggunaanBahanDetail::create([
                    'penggunaan_id' => $penggunaan->id,
                    'id_bahan' => $bahanId,
                    'jumlah' => $jumlah,
                    'satuan' => $r->satuan[$i] ?? 'pcs',
                    'keterangan' => $r->keterangan_detail[$i] ?? null
                ]);

                BahanPraktik::where('id', $bahanId)
                    ->decrement('stok', $jumlah);
            }

            DB::commit();
            return redirect()->route('siswa.penggunaan_bahan.index')
                ->with('success', 'Penggunaan bahan berhasil dicatat.');

        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', $e->getMessage());
        }
    }

    public function detail($id)
    {
        if (!session('siswa_id')) return redirect()->route('login');

        $penggunaan = PenggunaanBahan::with('detail.bahan','guru')
            ->where('id', $id)
            ->where('id_siswa', session('siswa_id'))
            ->firstOrFail();

        return view('siswa.penggunaan_bahan.detail', compact('penggunaan'));
    }
}
